
#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Transactions;
using System.Collections;
using System.Data;
using System.Linq;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
using gov.va.med.vbecs.Common;
using System.Collections.Generic;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
    [TestFixture]
    public class BOL_PendingBloodUnitTest : BaseTest
    {
        [SetUp]
        protected void SetUp()
        {
            this.BaseSetUp();
        }

        #region Get Pending Tests

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void OrderableTestTableToEnumCompare_Pass()
        {
            // Defect 286852
            // Arrange            
            var dt = UnitTestUtil.RunSQLGetDataTable("SELECT OrderableTestId from OrderableTest");
            var expected = dt.Rows.Count;
            var orderableTestEnums = Enum.GetValues(typeof(OrderableTest)).Cast<OrderableTest>();

            // Act
            var actual = 0;

            foreach (DataRow row in dt.Rows)
            {
                var orderableTestId = int.Parse(row["OrderableTestId"].ToString());

                if (orderableTestEnums.Any(x => (int)x == orderableTestId))
                {
                    actual++;
                }
                else
                {
                    // ERROR - This value was not found in the OrderableTest enums!
                }
            }

            // Assert all values in the database exist as an OrderableTest enum
            Assert.IsTrue(expected == actual);
            Assert.IsTrue(expected == orderableTestEnums.Count());
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AnitigenTypingOrderableTestsCount_Pass()
        {
            // Defect 286852
            // Arrange
            var expected = 114;

            // Act
            var actual = 0;
            var orderableTestEnums = Enum.GetValues(typeof(OrderableTest)).Cast<OrderableTest>();

            foreach (var orderableTest in orderableTestEnums)
            {
                if (PendingTestHelper.IsAntigenTypingOrderableTest(orderableTest))
                {
                    actual++;
                }
            }

            // Assert
            Assert.IsTrue(expected == actual);
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_False_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();

                // Act
                var actual = PendingBloodUnitTest.DoTestsExist(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_PendingStatus_True_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var actual = PendingBloodUnitTest.DoTestsExist(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_PendingStatus_False_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();
                InsertUnitAboRhData(PendingTestStatus.Accepted, 'V');

                // Act
                var actual = PendingBloodUnitTest.DoTestsExist(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void GetPendingBloodUnitTests_PendingStatus_NoData_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();
                InsertUnitAboRhData(PendingTestStatus.Accepted, 'V');

                // Act
                var data = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);
                var actual = data.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void GetPendingBloodUnitTests_PendingStatus_Data_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var data = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);
                var actual = data.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void GetValidBloodUnits_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var data = PendingBloodUnitTest.GetValidBloodUnits(this.TestDivisionCode, bloodUnitId, null);
                var actual = data.Count() > 0;

                // Assert
                Assert.IsTrue(actual);
            }
        }
        #endregion

        #region Reject Pending Tests

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without specifying a rejection comment
        public void RejectTest_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);
                PendingBloodUnitTest.RejectTests(pendingTests);
                var rejectedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Rejected);
                var actual = rejectedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void RejectTest_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.ReviewComment = "reject comment - test";
                }

                PendingBloodUnitTest.RejectTests(pendingTests);
                var rejectedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Rejected);
                var actual = rejectedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }
        #endregion

        #region Accept Pending Tests
        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_No_PendingTests_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = false;

                bool isWorkLoadDefined = false;
                var actual = PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, null, ref isWorkLoadDefined);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without a BloodUnitModel
        public void AcceptTest_AboRh_Null_BloodUnitModel_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    // Blood unit is required so this should make accepting fail
                    pendingTest.BloodUnitModel = null;
                }

                bool isWorkLoadDefined = false;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without an exception override comment
        public void AcceptTest_AboRh_NoExceptionOverride_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'W');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                bool isWorkLoadDefined = false;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without an exception override comment
        public void AcceptTest_AboRh_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'W');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    if (pendingTest.TestValidation != null &&
                        pendingTest.TestValidation.ExceptionForDisplayList != null)
                    {
                        foreach (var error in pendingTest.TestValidation.ExceptionForDisplayList)
                        {
                            // Exceptions require an override messag to accept so 
                            // this should cause an exception
                            error.OverrideMessage = null;
                        }
                    }
                }

                bool isWorkLoadDefined = false;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_With_OverrideException_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    if (pendingTest.TestValidation != null &&
                        pendingTest.TestValidation.ExceptionForDisplayList != null)
                    {
                        foreach (var error in pendingTest.TestValidation.ExceptionForDisplayList)
                        {
                            // Exceptions require an override messag to accept so 
                            error.OverrideMessage = "Unit Test";
                        }
                    }
                }

                bool isWorkLoadDefined = true;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected with TestValidationStatus == Error
        public void AcceptTest_AboRh_Error_Validation_Fail()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'E');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                bool isWorkLoadDefined = false;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_Skip_Anti_Dc_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'V');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestResultList.Add(new PendingTestResultModel(0, TestType.Anti_Dc_AHG, "Anti Dc", "1", "1+", null));

                    if (pendingTest.TestValidation != null &&
                        pendingTest.TestValidation.ExceptionForDisplayList != null)
                    {
                        foreach (var error in pendingTest.TestValidation.ExceptionForDisplayList)
                        {
                            // Exceptions require an override messag to accept so 
                            error.OverrideMessage = "Unit Test";
                        }
                    }
                }

                bool isWorkLoadDefined = false;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_Quarantine_Pass()
        {
            using (TransactionScope scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var bloodUnitId = InsertUnitAboRhData(PendingTestStatus.Pending, 'W');

                // Act
                var pendingTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Pending);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.ReviewComment = "AcceptTest_AboRh_Quarantine_Pass";

                    if (pendingTest.TestValidation != null &&
                        pendingTest.TestValidation.ExceptionForDisplayList != null)
                    {
                        foreach (var error in pendingTest.TestValidation.ExceptionForDisplayList)
                        {
                            // Exceptions require an override messag to accept so 
                            error.OverrideMessage = "Unit Test";
                        }
                    }
                }
                bool isWorkLoadDefined = true;
                PendingBloodUnitTest.AcceptTests(this.TestDivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingBloodUnitTest.GetPendingBloodUnitTests(this.TestDivisionCode, COMMON.PendingTestStatus.Accepted);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }
        #endregion

        private string InsertUnitAboRhData(PendingTestStatus pendingTestStatus, char interpState)
        {
            Guid rbcGuid = DataCreator.GetRandomBloodProduct(false, COMMON.ComponentClass.RBC).BloodProductGuid;
            BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, rbcGuid, ABO.A, RH.Positive, this.TestDivisionCode, false);

            //Default to Valid values..
            string antia15 = "1", antib16 = "0", antiab17 = "1", abointerp18 = "A", antid19 = "1", dcontrol20 = "0", rhinterp21="P";
            
            //I set the unit to APos to make this easier..TODO: someday create a fuller random method for data creation
            switch (interpState)
            {
                case 'W':
                    antia15 = "0";
                    antib16 = "1";
                    antiab17 = "1";
                    abointerp18 = "B";
                    antid19 = "1";
                    dcontrol20 = "1";
                    rhinterp21 = "I";
                    break;
                case 'E':
                    antia15 = "0";
                    antib16 = "0";
                    antiab17 = "0";
                    abointerp18 = "A";
                    antid19 = "0";
                    dcontrol20 = "0";
                    rhinterp21 = "P";
                    break;
            }

            var statusId = ((int)pendingTestStatus).ToString();
            var bloodUnitId = tmpUnit.LabelUnitId;
            var bloodUnitGuid = tmpUnit.BloodUnitGuid.ToString();
            string sql =
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 15, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + antia15 +"', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 16, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + antib16 + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 17, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + antiab17  + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 18, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + abointerp18 + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 19, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + antid19 + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 20, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + dcontrol20 + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')" +
            "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) VALUES (NULL, N'" + bloodUnitGuid + "', 21, N'" + bloodUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + rhinterp21 + "', N'instrument comment', NULL, NULL, N'ProVue', N'589  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', 81, N'714fad8e-0691-4720-a67f-0017dbb6bb44')";
            UnitTestUtil.RunSQL(sql);

            DataCreator.CreateWorkloadProcess(WorkloadProcessID.AutomatedUnitAbo);
            DataCreator.CreateWorkloadProcess(WorkloadProcessID.AutomatedUnitAboRh);
            return bloodUnitId;
        }

        private void DeleteTableData()
        {
            UnitTestUtil.RunSQL("DELETE FROM PendingBloodUnitTest");
        }
    }
}


#endif



